<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\AllUnitsType;
use App\Models\UnitsStatus;
use App\Models\Unit;
use App\Models\UnitsNotification;
use App\Models\UnitsBoard;
use App\Models\UnitsBoardsFile;
use App\Models\UnitsPicture;
use App\Models\UnitsFile;
use App\Models\UnitsFilesFolder;
use App\Models\UnitsStaffFile;
use App\Models\UnitsManagersFile;
use App\Models\UnitsOwnersFile;
use App\Models\UnitsMeasurement;
use App\Models\UnitsMeasurementsPicture;
use App\Models\UnitsMeasurementsFile;
use App\Models\UnitsMaterial;
use App\Models\UnitsMaterialsPicture;
use App\Models\UnitsMaterialsFile;
use App\Models\UnitsLocation;
use App\Models\PicturesFolder;
use App\Models\PropertiesParkingsSpacesHistoryLogs;
use App\Models\PropertiesParkingSpace;
use App\Models\UnitsNote;
use App\Models\UnitsNotesFile;
use App\Models\UnitsFeature;
use App\Models\UnitsUtility;



use App\Models\UsersRole;
use App\Models\User;
use Carbon\Carbon;

use App\Libraries\Generic;


class UnitsController extends Controller
{
    
    protected $generic;
    
    public function __construct(){
        
        $this->generic = new Generic();
        
    }

    public function unit(Request $request){
        
        // echo "<pre>";
        // print_r($request->all());
        // exit();

        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);

        // check if permission to do
        if(!User::checkPermission('add-edit-unit')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'type_id' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }

        $update_criteria = [];
        if(!empty($request->id)){
            $update_criteria['id'] = $request->id;
        }
        else{
            $update_criteria['id'] = 0;
        }
        
        // echo "<pre>";
        // print_r( $request->all());
        // exit();

        $unit = Unit::updateOrCreate(
            $update_criteria, 
            [
                'name' => $request->name ?? '',
                'unique_id' => 'U-'.random_int(1000000000, 9999999999),
                'is_active' => $request->is_active ?? 1,
                'property_id' => $request->property_id  ?? null,
                'floor' => $request->floor  ?? 0,
                'unit_status_id' => $request->unit_status_id  ?? 1,
                'type_id' => $request->type_id  ?? 1,
                'bedrooms' => $request->bedrooms  ?? 0,
                'bathrooms' => $request->bathrooms  ?? 0,
                'stairs_to_unit' => $request->stairs_to_unit  ?? 0,
                'stairs_in_unit' => $request->stairs_in_unit  ?? 0,
                'elevator_to_unit' => $request->elevator_to_unit  ?? 0,
                'elevator_in_unit' => $request->elevator_in_unit  ?? 0,
                'monthly_rent' => $request->monthly_rent  ?? '',
                'security_deposit' => $request->security_deposit  ?? '',
                'last_month_rent_required' => $request->last_month_rent_required  ?? 0,
                'couples_allowed' => $request->couples_allowed  ?? 0,
                'childern_allowed' => $request->childern_allowed  ?? 0,
                'pets_allowed' => $request->pets_allowed  ?? 0,
                'pets_deposit' => $request->pets_deposit  ?? 0,
                'pets_rent' => $request->pets_rent  ?? 0,
                'pets_more_info' => $request->pets_more_info  ?? '',
                'smoking_permission_id' => $request->smoking_permission_id ?? null,
                'lgbtq_friendly' => $request->lgbtq_friendly  ?? 0,
                'gender_preference_id' => $request->gender_preference_id  ?? 1,
                'wifi_network' => $request->wifi_network ?? '',
                'wifi_password' => $request->wifi_password ?? '',
                'alarm_code' => $request->alarm_code ?? '',
                'gate_code' => $request->gate_code ?? '',
                'door_code' => $request->door_code ?? '',
                'mailbox_info' => $request->mailbox_info ?? '',
                'other_info' => $request->other_info ?? '',
                'picture_template_id' => $request->picture_template_id ?? null,
                'file_template_id' => $request->file_template_id ?? null,
                'is_parking_space_assigned' => $request->is_parking_space_assigned ?? 0,
                'property_parking_space_id' => $request->property_parking_space_id ?? null,
                'embed' => $request->embed ?? '',
                'listing_title' => $request->listing_title ?? '',
                'listing_description' => $request->listing_description ?? '',
                'other_unit_features' => $request->other_unit_features ?? '',
                'owner_id' => $owner_id,
                'creator_id' => $user_id
            ]
        );
        
        $unit_id = $unit->id;
        
        // unit locations
        $unit_locations = $request->unit_locations ?? [];
        
        foreach($unit_locations as $ul){
            
            $update_criteria = [];
            if(!empty($ul['id'])){
                $update_criteria['id'] = $ul['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            $unit_location = UnitsLocation::updateOrCreate($update_criteria, [
                'location' => $ul['location'] ?? '',
                'description' => $ul['description'] ?? '',
                'owner_id' => $owner_id,
                'unit_id' => $unit_id
            ]);
            
            $unit_location_id = $unit_location->id;
            
        }
        
        // notifications
        $notifications = $request->notifications ?? [];

        foreach($notifications as $n){
            
            $update_criteria = [];
            if(!empty($n['id'])){
                $update_criteria['id'] = $n['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            $date = (!empty($n['posting_date'])) ? $this->generic->formatInputDate($n['posting_date']) : date('d-m-Y');
            $time = $n['posting_time'] ?? date('h:m:i A');
            $str = strtotime($date.' '.$time);
            
            UnitsNotification::updateOrCreate($update_criteria, [
                'title' => $n['title'] ?? '',
                'notification' => $n['notification'] ?? '',
                'is_popup_notification' => $n['is_popup_notification'] ?? 0,
                'posting_date' => $date,
                'posting_time' => $time,
                'posting_date_time_str' => $str,
                'unit_id' => $unit_id
            ]);
            
        }
        
        
        // boards
        $boards = $request->boards ?? [];

        foreach($boards as $b){
            
            $update_criteria = [];
            if(!empty($b['id'])){
                $update_criteria['id'] = $b['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            $boards = UnitsBoard::updateOrCreate($update_criteria, [
                'unit_id' => $unit_id,
                'board_type_id' => $b['board_type_id'] ?? null,
                'title' => $b['title']  ?? '',
                'icon' => $b['icon'] ?? '',
                'board' => $b['board'] ?? 0,
                'link' => $b['link'] ?? '',
                'link_show_in' => $b['link_show_in'] ?? 0,
                'order_number' => $b['order_number'] ?? 1
            ]);
            
            $board_id = $board->id;
            
            $files = $b['files'];
            
            foreach($files as $f){
                
                $update_criteria = [];
                if(!empty($p['id'])){
                    $update_criteria['id'] = $f['id'];
                }
                else{
                    $update_criteria['id'] = 0;
                }
                
                PropertiesBoardsFile::updateOrCreate($update_criteria, [
                    'property_board_id' => $board_id,
                    'title' => $f['title'] ?? '',
                    'file' => $f['file']
                ]);
                
            }
            
        }
        
        // pictures
        $pictures = $request->pictures ?? [];

        foreach($pictures as $p){
            
            $update_criteria = [];
            if(!empty($p['id'])){
                $update_criteria['id'] = $p['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            UnitsPicture::updateOrCreate($update_criteria, [
                'picture' => $p['picture'],
                'title' => $p['title'] ?? '',
                'caption' => $p['caption'] ?? '',
                'is_cover' => $p['is_cover'] ?? 0,
                'order_number' => $p['order_number'] ?? 1,
                'unit_id' => $unit_id,
                'picture_folder_id' => $p['picture_folder_id'] ?? null
            ]);
        }
        
        // files
        $files = $request->unit_files ?? [];
        
        foreach($files as $f){
            
            $update_criteria = [];
            if(!empty($f['id'])){
                $update_criteria['id'] = $f['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            UnitsFile::updateOrCreate($update_criteria, [ 
                'title' => $f['title'] ?? '',
                'file' => $f['file'],
                'unit_id' => $unit_id,
                'unit_file_folder_id ' => $f['unit_file_folder_id'] ?? null,
                'property_manager_sharing' => $f['property_manager_sharing'] ?? 0,
                'staff_sharing' => $f['staff_sharing'] ?? 0,
                'tenant_sharing' => $f['tenant_sharing'] ?? 0,
            ]);
            
        }
        
        // staff files
        $staff_files = $request->unit_staff_files ?? [];

        foreach($staff_files as $f){
            
            $update_criteria = [];
            if(!empty($f['id'])){
                $update_criteria['id'] = $f['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            UnitsStaffFile::updateOrCreate($update_criteria, [
                'unit_id' => $unit_id,
                'title' => $f['title'] ?? '',
                'file' => $f['file']
            ]);
            
        }
        
        // managers files
        $manager_files = $request->unit_manager_files ?? [];

        foreach($manager_files as $f){
            
            $update_criteria = [];
            if(!empty($f['id'])){
                $update_criteria['id'] = $f['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            UnitsManagersFile::updateOrCreate($update_criteria, [
                'unit_id' => $unit_id,
                'title' => $f['title'] ?? '',
                'file' => $f['file']
            ]);
            
        }
        
        // owner files
        $owner_files = $request->unit_owner_files ?? [];

        foreach($owner_files as $f){
            
            $update_criteria = [];
            if(!empty($f['id'])){
                $update_criteria['id'] = $f['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            UnitsOwnersFile::updateOrCreate($update_criteria, [
                'unit_id' => $unit_id,
                'title' => $f['title'] ?? '',
                'file' => $f['file']
            ]);
            
        }
        
        // units public notes
        $notes = $request->unit_notes ?? [];

        foreach($notes as $n){
            
            $update_criteria = [];
            if(!empty($n['id'])){
                $update_criteria['id'] = $n['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            $note = UnitsNote::updateOrCreate($update_criteria, [
                'unit_id' => $unit_id,
                'title' => $n['title'] ?? '',
                'note' => $n['note'] ?? '',
                'property_manager_sharing' => $n['property_manager_sharing'] ?? 'hidden',
                'staff_sharing' => $n['staff_sharing'] ?? 'hidden',
                'tenant_sharing' => $n['tenant_sharing'] ?? 'hidden'
            ]);
            
            $note_id = $note->id;
            
            $files = $n['files'];
            
            foreach($files as $f){
                
                $update_criteria = [];
                if(!empty($p['id'])){
                    $update_criteria['id'] = $f['id'];
                }
                else{
                    $update_criteria['id'] = 0;
                }
                
                UnitsNotesFile::updateOrCreate($update_criteria, [
                    'unit_note_id ' => $note_id,
                    'title' => $f['title'] ?? '',
                    'file' => $f['file']
                ]);
                
            }
        }
        
        // // units staff notes
        // $notes = $request->unit_staff_notes ?? [];

        // foreach($notes as $n){
            
        //     $update_criteria = [];
        //     if(!empty($n['id'])){
        //         $update_criteria['id'] = $n['id'];
        //     }
        //     else{
        //         $update_criteria['id'] = 0;
        //     }
            
        //     UnitsNote::updateOrCreate($update_criteria, [
        //         'unit_id' => $unit_id,
        //         'note' => $n['note'] ?? '',
        //         'type' => 'staff',
        //     ]);
        // }
        
        // // units manager notes
        // $notes = $request->unit_manager_notes ?? [];

        // foreach($notes as $n){
            
        //     $update_criteria = [];
        //     if(!empty($n['id'])){
        //         $update_criteria['id'] = $n['id'];
        //     }
        //     else{
        //         $update_criteria['id'] = 0;
        //     }
            
        //     UnitsNote::updateOrCreate($update_criteria, [
        //         'unit_id' => $unit_id,
        //         'note' => $n['note'] ?? '',
        //         'type' => 'manager',
        //     ]);
        // }
        
        // // units owner notes
        // $notes = $request->unit_owner_notes ?? [];

        // foreach($notes as $n){
            
        //     $update_criteria = [];
        //     if(!empty($n['id'])){
        //         $update_criteria['id'] = $n['id'];
        //     }
        //     else{
        //         $update_criteria['id'] = 0;
        //     }
            
        //     UnitsNote::updateOrCreate($update_criteria, [
        //         'unit_id' => $unit_id,
        //         'note' => $n['note'] ?? '',
        //         'type' => 'owner',
        //     ]);
        // }
        
        // measurements
        $measurements = $request->measurements ?? [];

        foreach($measurements as $m){
            
            $update_criteria = [];
            if(!empty($m['id'])){
                $update_criteria['id'] = $m['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            $measurement = UnitsMeasurement::updateOrCreate($update_criteria, [
                'name' => $m['name'],
                'unit_id' => $unit_id,
                'measurement_type_id' => $m['measurement_type_id'],
                'unit_location_id' => $m['unit_location_id'],
                'measurement_unit_id' => $m['measurement_unit_id'],
                'length' => $m['length'] ?? "",
                'width' => $m['width'] ?? "",
                'height' => $m['height'] ?? "",
                'depth' => $m['depth'] ?? "",
                'notes' => $m['notes'] ?? ""
            ]);
            
            $unit_measurement_id = $measurement->id;
            
            $pictures = $m['pictures'];
            
            foreach($pictures as $p){
                
                $update_criteria = [];
                if(!empty($p['id'])){
                    $update_criteria['id'] = $p['id'];
                }
                else{
                    $update_criteria['id'] = 0;
                }
                
                UnitsMeasurementsPicture::updateOrCreate($update_criteria, [
                    'unit_measurement_id' => $unit_measurement_id,
                    'picture' => $p['picture'],
                    'order_number' => $p['order_number']
                ]);
                
            }
            
            $files = $m['files'];
            
            foreach($files as $f){
                
                $update_criteria = [];
                if(!empty($p['id'])){
                    $update_criteria['id'] = $f['id'];
                }
                else{
                    $update_criteria['id'] = 0;
                }
                
                UnitsMeasurementsFile::updateOrCreate($update_criteria, [
                    'unit_measurement_id' => $unit_measurement_id,
                    'title' => $f['title'] ?? '',
                    'file' => $f['file']
                ]);
                
            }
            
        }
        
        // materials
        $materials = $request->materials ?? [];

        foreach($materials as $m){
            
            $update_criteria = [];
            if(!empty($m['id'])){
                $update_criteria['id'] = $m['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            $material = UnitsMaterial::updateOrCreate($update_criteria, [
                'unit_id' => $unit_id,
                'material_type_id' => $m['material_type_id'],
                'material_id' => $m['material_id'],
                'unit_location_id' => $m['unit_location_id'],
                'last_installed' => $m['last_installed'] ?? '',
                'installed_by' => null,
                'install_notes' => $m['install_notes'] ?? ''
            ]);
            
            $unit_material_id = $material->id;
            
            $pictures = $m['pictures'];
            
            foreach($pictures as $p){
                
                $update_criteria = [];
                if(!empty($p['id'])){
                    $update_criteria['id'] = $p['id'];
                }
                else{
                    $update_criteria['id'] = 0;
                }
                
                UnitsMaterialsPicture::updateOrCreate($update_criteria, [
                    'unit_material_id' => $unit_material_id,
                    'picture' => $p['picture'],
                    'order_number' => $p['order_number']
                ]);
                
            }
            
            $files = $m['files'];
            
            foreach($files as $f){
                
                $update_criteria = [];
                if(!empty($p['id'])){
                    $update_criteria['id'] = $f['id'];
                }
                else{
                    $update_criteria['id'] = 0;
                }
                
                UnitsMaterialsFile::updateOrCreate($update_criteria, [
                    'unit_material_id' => $unit_material_id,
                    'title' => $f['title'] ?? '',
                    'file' => $f['file']
                ]);
                
            }
            
        }
        
        // utilities included with rent
        $utilities_included_with_rent = $request->utilities_included_with_rent ?? [];
        
        foreach($utilities_included_with_rent as $u){
            
            $update_criteria = [];
            if(!empty($u['id'])){
                $update_criteria['id'] = $u['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            UnitsUtility::updateOrCreate($update_criteria, [
                'unit_id' => $unit_id,
                'utility_id' => $u['utility_id'],
                'included_in_rent' => 1,
                'addition_to_rent' => 0,
            ]);
        }
        
        // utilities addition to rent
        $utilities_addition_to_rent = $request->utilities_addition_to_rent ?? [];
        
        foreach($utilities_addition_to_rent as $u){
            
            $update_criteria = [];
            if(!empty($u['id'])){
                $update_criteria['id'] = $u['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            UnitsUtility::updateOrCreate($update_criteria, [
                'unit_id' => $unit_id,
                'utility_id' => $u['utility_id'],
                'included_in_rent' => 0,
                'addition_to_rent' => 1,
            ]);
        }
        
        // features
        $features = $request->features ?? [];

        foreach($features as $f){
            
            $update_criteria = [];
            if(!empty($f['id'])){
                $update_criteria['id'] = $f['id'];
            }
            else{
                $update_criteria['id'] = 0;
            }
            
            UnitsFeature::updateOrCreate($update_criteria, [
                'unit_id' => $unit_id,
                'feature_id' => $f['feature_id']
            ]);
        }
        
        // parking space assignment work
        $is_parking_space_assigned = $request->is_parking_space_assigned ?? 0;
        $property_parking_space_id = $request->property_parking_space_id ?? null;
        
        if($is_parking_space_assigned == 1){    // update parking space tables accordingly
            $history_log = PropertiesParkingsSpacesHistoryLogs::where(['property_space_parking_id'=>$property_parking_space_id, 'unit_id'=>$unit_id])->first();
            
            if(empty($history_log)){    // if log is empty it means it is new assignment, so create history log
                PropertiesParkingsSpacesHistoryLogs::create([
                    'property_space_parking_id' => $property_parking_space_id,
                    'tenant_id' => null,
                    'unit_id' => $unit_id,
                    'start_date' => date('d-m-Y'),
                    'end_date' => ''
                ]);
                
                // now mark parking space as assigned
                PropertiesParkingSpace::where(['id'=>$property_parking_space_id])->update(['is_assigned'=>1]);
            }
        }
        
        return response()->json(['message' => 'Added Successfully', 'data'=> $unit_id ]);
        
    }

    public function getUnit(Request $request)
    {

        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);

        // check if permission to do
        if(!User::checkPermission('get-unit')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }

        $unit_id = $request->query('unit_id');
        
        $unit = Unit::find($unit_id);

        if(empty($unit)){
            return response()->json(['message' => 'No unit found']);
        }
        
        // check if have access to view or edit
        if($unit->owner_id != $owner_id){
            return response()->json(config('constants.messages.invalid_access'), 400);
            exit();
        }

        $unit = $unit->toArray();
        
        $unit['unit_locations'] =       Unit::find($unit_id)->unitLocations;
        $unit['notifications'] =        Unit::find($unit_id)->notifications;
        $unit['boards'] =               Unit::find($unit_id)->boards;
        $unit['pictures'] =             Unit::find($unit_id)->pictures;
        $unit['files'] =                Unit::find($unit_id)->files;
        $unit['staff_files'] =          Unit::find($unit_id)->staffFiles;
        $unit['managers_files'] =       Unit::find($unit_id)->managersFiles;
        $unit['owners_files'] =         Unit::find($unit_id)->ownersFiles;
        
        $unit['unit_notes'] =         Unit::allNotes($unit_id);
        
        // $unit['unit_notes'] =               UnitsNote::notes($unit_id);
        // $unit['unit_staff_notes'] =         UnitsNote::staffNotes($unit_id);
        // $unit['unit_managers_notes'] =      UnitsNote::managerNotes($unit_id);
        // $unit['unit_owners_notes'] =        UnitsNote::ownerNotes($unit_id);
        
        $unit['measurements'] =         UnitsMeasurement::getUnitMeasurements($unit_id);
        $unit['materials'] =            UnitsMaterial::getUnitMaterials($unit_id);
        $unit['utilities'] =            Unit::find($unit_id)->utilities;
        $unit['features'] =             Unit::find($unit_id)->features;
        
        return response()->json(['data' => $unit]);
    }

    public function getMyUnits()
    {
        
        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);

        // check if permission to do
        if(!User::checkPermission('get-my-units')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }
        
        $units = unit::select(['id', 'name', 'unique_id', 'is_active', 'property_id', 'unit_status_id'])->where('owner_id', $owner_id)->get()->toArray();
                
        return response()->json(['data' => $units]);
    }
    
    public function markUnitsAsActive(Request $request){
        
        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);
        
        // check if permission to do
        if(!User::checkPermission('add-edit-unit')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }
        
        $validator = Validator::make($request->all(), [
            'ids' => 'required'
        ]);
        
        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }
        
        $ids = $request->ids;
        
        foreach($ids as $id){
            Unit::where(['id'=>$id])->update(['is_active'=>1]);
        }
        
        return response()->json(['message' => 'Updated Successfully']);
        
    }
    
    public function markUnitsAsInactive(Request $request){
        
        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);
        
        // check if permission to do
        if(!User::checkPermission('add-edit-unit')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }
        
        $validator = Validator::make($request->all(), [
            'ids' => 'required'
        ]);
        
        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }
        
        $ids = $request->ids;
        
        foreach($ids as $id){
            Unit::where(['id'=>$id])->update(['is_active'=>0]);
        }
        
        return response()->json(['message' => 'Updated Successfully']);
        
    }

    public function getAllUnitsTypes(){

        $all_Units_types = AllUnitsType::select('id', 'type')->get()->toArray();

        return response()->json(['data' => $all_Units_types]);

    }
    
    public function getUnitsStatuses(){

        $unit_statuses = UnitsStatus::select('id', 'status')->get()->toArray();

        return response()->json(['data' => $unit_statuses]);

    }
    
    public function deleteUnit(Request $request){

        // check if permission to do
        if(!User::checkPermission('delete-unit')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }

        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);
        
        $unit_id = $request->query('unit_id');
        $unit = Unit::find($unit_id);
        
        // check if unit exists
        if(empty($unit)){
            return response()->json(['message' => 'No unit found']);
        }
        
        // check if have access to view or edit
        if($unit->owner_id != $owner_id){
            return response()->json(config('constants.messages.invalid_access'), 400);
            exit();
        }
        
        Unit::find($unit_id)->delete();
        
        return response()->json(['message' => 'Deleted', 'data'=> [] ]);

    }
    
    public function deleteUnits(Request $request){
        
        // check if permission to do
        if(!User::checkPermission('delete-units')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }
        
        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);
        
        
        $validator = Validator::make($request->all(), [
            'ids' => 'required'
        ]);
        
        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }
        
        $ids = $request->ids ?? [];
        
        foreach($ids as $unit_id){
            $unit = Unit::find($unit_id);
            
            if(!empty($unit)){
                if($unit->owner_id == $owner_id){
                    Unit::find($unit_id)->delete();
                }
            }
            
        }
        
        return response()->json(['message' => 'Deleted', 'data'=> [] ]);

    }
    
    public function duplicateUnit(Request $request)
    {
        
        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);
        
        // check if permission to do
        if(!User::checkPermission('duplicate-unit')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }
        
        $validator = Validator::make($request->all(), [
            'unit_id' => 'required|numeric',
            'number_of_units' => 'required|numeric'
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }
        
        $number_of_units = $request->number_of_units;
        
        $unit_id = $request->unit_id;
        $unit = Unit::find($unit_id);
        
        // check if unit exists
        if(empty($unit)){
            return response()->json(['message' => 'No unit found']);
        }
        
        // check if have access to view or edit
        if($unit->owner_id != $owner_id){
            return response()->json(config('constants.messages.invalid_access'), 400);
            exit();
        }
        
        for($q=0; $q<$number_of_units; $q++){
            
            // before start duplicating unit, duplicate picture folder template that this unit is using
            $new_picture_template_folders_maping = [];
            $new_picture_template_id = null;
            if(!empty($unit->picture_template_id)){
                $old_picture_template_id = $unit->picture_template_id;
                $new_picture_template_id = $this->generic->duplicatePictureTemplate($old_picture_template_id, '');
                
                $old_picture_template_folders = PicturesFolder::where(['picture_template_id'=>$old_picture_template_id])->get();
                $old_picture_template_folders = (!empty($old_picture_template_folders)) ? $old_picture_template_folders->toArray() : [];
                
                $new_picture_template_folders = PicturesFolder::where(['picture_template_id'=>$new_picture_template_id])->get();
                $new_picture_template_folders = (!empty($new_picture_template_folders)) ? $new_picture_template_folders->toArray() : [];
                
                for($i=0; $i<count($new_picture_template_folders); $i++){
                    $new_picture_template_folders_maping[$old_picture_template_folders[$i]['id']] = $new_picture_template_folders[$i]['id'];
                }
            }
            
            // before start duplicating unit, duplicate file folder template that this unit is using
            $new_file_template_folders_maping = [];
            $new_file_template_id = null;
            if(!empty($unit->file_template_id)){
                $old_file_template_id = $unit->file_template_id;
                $new_file_template_id = $this->generic->duplicateFileTemplate($old_file_template_id, '');
                
                $old_file_template_folders = FilesFolder::where(['file_template_id'=>$old_file_template_id])->get();
                $old_file_template_folders = (!empty($old_file_template_folders)) ? $old_file_template_folders->toArray() : [];
                
                $new_file_template_folders = FilesFolder::where(['file_template_id'=>$new_file_template_id])->get();
                $new_file_template_folders = (!empty($new_file_template_folders)) ? $new_file_template_folders->toArray() : [];
                
                for($i=0; $i<count($new_file_template_folders); $i++){
                    $new_file_template_folders_maping[$old_file_template_folders[$i]['id']] = $new_file_template_folders[$i]['id'];
                }
            }
            
            // unt basic info
            $new_unit = Unit::create(
                [
                    'name' => $unit->name." copy ".($q+1) ?? '',
                    'unique_id' => 'U-'.random_int(1000000000, 9999999999),
                    'is_active' => $unit->is_active ?? 1,
                    'property_id' => $unit->property_id  ?? null,
                    'floor' => $unit->floor  ?? 0,
                    'unit_status_id' => $unit->unit_status_id  ?? 1,
                    'type_id' => $unit->type_id  ?? 1,
                    'bedrooms' => $unit->bedrooms  ?? 0,
                    'bathrooms' => $unit->bathrooms  ?? 0,
                    'stairs_to_unit' => $unit->stairs_to_unit  ?? 0,
                    'stairs_in_unit' => $unit->stairs_in_unit  ?? 0,
                    'elevator_to_unit' => $unit->elevator_to_unit  ?? 0,
                    'elevator_in_unit' => $unit->elevator_in_unit  ?? 0,
                    'monthly_rent' => $unit->monthly_rent  ?? '',
                    'security_deposit' => $unit->security_deposit  ?? '',
                    'last_month_rent_required' => $unit->last_month_rent_required  ?? 0,
                    'couples_allowed' => $unit->couples_allowed  ?? 0,
                    'childern_allowed' => $unit->childern_allowed  ?? 0,
                    'pets_allowed' => $unit->pets_allowed  ?? 0,
                    'pets_deposit' => $unit->pets_deposit  ?? 0,
                    'pets_rent' => $unit->pets_rent  ?? 0,
                    'pets_more_info' => $unit->pets_more_info  ?? '',
                    'smoking_permission_id' => $unit->smoking_permission_id ?? null,
                    'lgbtq_friendly' => $unit->lgbtq_friendly  ?? 0,
                    'gender_preference_id' => $unit->gender_preference_id  ?? 1,
                    'wifi_network' => $unit->wifi_network ?? '',
                    'wifi_password' => $unit->wifi_password ?? '',
                    'alarm_code' => $unit->alarm_code ?? '',
                    'gate_code' => $unit->gate_code ?? '',
                    'door_code' => $unit->door_code ?? '',
                    'mailbox_info' => $unit->mailbox_info ?? '',
                    'other_info' => $unit->other_info ?? '',
                    'picture_template_id' => $new_picture_template_id,
                    'file_template_id' => $new_file_template_id,
                    'is_parking_space_assigned' => $unit->is_parking_space_assigned ?? 0,
                    'property_parking_space_id' => $unit->property_parking_space_id ?? null,
                    'embed' => $unit->embed ?? '',
                    'listing_title' => $unit->listing_title ?? '',
                    'listing_description' => $unit->listing_description ?? '',
                    'other_unit_features' => $unit->other_unit_features ?? '',
                    'owner_id' => $owner_id,
                    'creator_id' => $user_id
                ]
            );
            
            $new_unit_id = $new_unit->id;
            
            // unit location
            $unit_locations = UnitsLocation::where(['unit_id'=>$unit_id])->get();
            $unit_locations = (!empty($unit_locations)) ? $unit_locations : [];
            
            foreach($unit_locations as $ul){
                
                $unit_location = UnitsLocation::create([
                    'location' => $ul['location'],
                    'description' => $ul['description'],
                    'owner_id' => $owner_id,
                    'unit_id' => $new_unit_id
                ]);
                
                $unit_location_id = $unit_location->id;
                
            }
            
            // notifications
            $notifications = Unit::find($unit_id)->notifications;
            $notifications = (!empty($notifications)) ? $notifications : [];
    
            foreach($notifications as $n){
                
                UnitsNotification::create([
                    'title' => $n['title'],
                    'notification' => $n['notification'],
                    'is_popup_notification' => $n['is_popup_notification'],
                    'posting_date' => $n['posting_date'],
                    'posting_time' => $n['posting_time'],
                    'posting_date_time_str' => $n['posting_date_time_str'],
                    'unit_id' => $new_unit_id
                ]);
                
            }
            
            // boards
            $boards = Unit::find($unit_id)->boards;
            $boards = (!empty($boards)) ? $boards : [];
    
            foreach($boards as $b){
                
                UnitsBoard::create([
                    'unit_id' => $new_unit_id,
                    'board_type_id' => $b['board_type_id'],
                    'title' => $b['title'],
                    'icon' => $b['icon'],
                    'board' => $b['board'],
                    'link' => $b['link'],
                    'link_show_in' => $b['link_show_in'],
                    'order_number' => $b['order_number']
                ]);
                
            }
            
            // pictures
            $pictures = Unit::find($unit_id)->pictures;
            $pictures = (!empty($pictures)) ? $pictures : [];
    
            foreach($pictures as $p){
                
                UnitsPicture::create([
                    'picture' => $p['picture'],
                    'title' => $p['title'],
                    'caption' => $p['caption'],
                    'is_cover' => $p['is_cover'],
                    'order_number' => $p['order_number'],
                    'unit_id' => $new_unit_id,
                    'picture_folder_id' => (isset($new_picture_template_folders_maping[$p['picture_folder_id']])) ? $new_picture_template_folders_maping[$p['picture_folder_id']] : null,
                ]);
            }
            
            // unit files
            $files = Unit::find($unit_id)->files;
            $files = (!empty($files)) ? $files : [];
    
            foreach($files as $f){
                UnitsFile::create([
                    'title' => $f['title'] ?? '',
                    'file' => $f['file'],
                    'unit_id' => $new_unit_id,
                    'file_folder_id' => (isset($new_file_template_folders_maping[$f['file_folder_id']])) ? $new_file_template_folders_maping[$f['file_folder_id']] : null,
                ]);
            }
            
            // staff files
            $staff_files = Unit::find($unit_id)->staffFiles;
            $staff_files = (!empty($staff_files)) ? $staff_files : [];
    
            foreach($staff_files as $f){
                UnitsStaffFile::create([
                    'unit_id' => $new_unit_id,
                    'title' => $f['title'] ?? '',
                    'file' => $f['file']
                ]);
            }
            
            // managers files
            $manager_files = Unit::find($unit_id)->managersFiles;
            $manager_files = (!empty($manager_files)) ? $manager_files : [];
    
            foreach($manager_files as $f){
                UnitsManagersFile::create([
                    'unit_id' => $new_unit_id,
                    'title' => $f['title'] ?? '',
                    'file' => $f['file']
                ]);
            }
            
            // owner files
            $owner_files = Unit::find($unit_id)->ownersFiles;
            $owner_files = (!empty($owner_files)) ? $owner_files : [];
    
            foreach($owner_files as $f){
                UnitsOwnersFile::create([
                    'unit_id' => $new_unit_id,
                    'title' => $f['title'] ?? '',
                    'file' => $f['file']
                ]);
            }
            
            // all notes
            $notes = Unit::find($unit_id)->allNotes;
            $notes = (!empty($notes)) ? $notes : [];
    
            foreach($notes as $n){
                UnitsNote::create([
                    'unit_id' => $new_unit_id,
                    'title' => $n['title'] ?? '',
                    'note' => $n['note'] ?? '',
                    'property_manager_sharing' => $n['property_manager_sharing'] ?? 'hidden',
                    'staff_sharing' => $n['staff_sharing'] ?? 'hidden',
                    'tenant_sharing' => $n['tenant_sharing'] ?? 'hidden'
                ]);
            }
            
            // measurements
            $measurements = UnitsMeasurement::getUnitMeasurements($unit_id);
            $measurements = (!empty($measurements)) ? $measurements : [];
    
            foreach($measurements as $m){
                
                $measurement = UnitsMeasurement::create([
                    'name' => $m['name'],
                    'unit_id' => $new_unit_id,
                    'measurement_type_id' => $m['measurement_type_id'],
                    'unit_location_id' => $m['unit_location_id'],
                    'measurement_unit_id' => $m['measurement_unit_id'],
                    'length' => $m['length'] ?? "",
                    'width' => $m['width'] ?? "",
                    'height' => $m['height'] ?? "",
                    'depth' => $m['depth'] ?? "",
                    'notes' => $m['notes'] ?? ""
                ]);
                
                $unit_measurement_id = $measurement->id;
                
                $pictures = $m['pictures'];
                
                foreach($pictures as $p){
                    UnitsMeasurementsPicture::create([
                        'unit_measurement_id' => $unit_measurement_id,
                        'picture' => $p['picture'],
                        'order_number' => $p['order_number']
                    ]);
                }
                
                $files = $m['files'];
                
                foreach($files as $f){
                    UnitsMeasurementsFile::create([
                        'unit_measurement_id' => $unit_measurement_id,
                        'title' => $f['title'] ?? '',
                        'file' => $f['file']
                    ]);
                }
                
            }
            
            // materials
            $materials = UnitsMaterial::getUnitMaterials($unit_id);
            $materials = (!empty($materials)) ? $materials : [];
    
            foreach($materials as $m){
                
                $material = UnitsMaterial::create([
                    'unit_id' => $new_unit_id,
                    'material_type_id' => $m['material_type_id'],
                    'material_id' => $m['material_id'],
                    'unit_location_id' => $m['unit_location_id'],
                    'last_installed' => $m['last_installed'] ?? '',
                    'installed_by' => $m['installed_by'] ?? null,
                    'install_notes' => $m['install_notes'] ?? ''
                ]);
                
                $unit_material_id = $material->id;
                
                $pictures = $m['pictures'];
                
                foreach($pictures as $p){
                    
                    UnitsMaterialsPicture::create([
                        'unit_material_id' => $unit_material_id,
                        'picture' => $p['picture'],
                        'order_number' => $p['order_number']
                    ]);
                    
                }
                
                $files = $m['files'];
                
                foreach($files as $f){
                    
                    UnitsMaterialsFile::create([
                        'unit_material_id' => $unit_material_id,
                        'title' => $f['title'] ?? '',
                        'file' => $f['file']
                    ]);
                    
                }
                
            }
            
            // utilities
            $utilities = Unit::find($unit_id)->utilities;
            
            foreach($utilities as $u){
                
                UnitsUtility::create([
                    'unit_id' => $new_unit_id,
                    'utility_id' => $u['utility_id'],
                    'included_in_rent' => $u['included_in_rent'],
                    'addition_to_rent' => $u['addition_to_rent']
                ]);
                
            }
            
            // features
            $features = Unit::find($unit_id)->features;
            
            foreach($features as $f){
                
                UnitsFeature::create([
                    'unit_id' => $new_unit_id,
                    'feature_id' => $f['feature_id']
                ]);
                
            }
            
        }
        
        
        
        return response()->json(['message' => 'success', 'data' => []]);
    }
    
    public function inlineChangeUnitName(Request $request)
    {
        
        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);
        
        // check if permission to do
        if(!User::checkPermission('inline-change-unit-name')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }
        
        $validator = Validator::make($request->all(), [
            'id' => 'required|numeric',
            'name' => 'required'
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }
        
        $name = $request->name;
        
        $unit_id = $request->id;
        $unit = Unit::find($unit_id);
        
        // check if unit exists
        if(empty($unit)){
            return response()->json(['message' => 'No unit found']);
        }
        
        // check if have access to view or edit
        if($unit->owner_id != $owner_id){
            return response()->json(config('constants.messages.invalid_access'), 400);
            exit();
        }
        
        $data = [
            'name' => $name
        ];
        
        Unit::where(['id'=>$unit_id])->update($data);
        
        return response()->json(['message' => 'success', 'data' => []]);
    }

    
    public function rearrangeUnitBoards(Request $request){
        
        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);
        
        // check if permission to do
        if(!User::checkPermission('rearrange-unit-boards')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }
        
        $validator = Validator::make($request->all(), [
            'unit_id' => 'required',
            'data' => 'required'
        ]);
        
        if ($validator->fails()) { 
            return response()->json($validator->errors(), 400);
        }
        
        $unit_id = $request->unit_id;
        $data = $request->data;
        
        $unit = Unit::find($unit_id);
        
        // check if unit exists
        if(empty($unit)){
            return response()->json(['message' => 'No unit found']);
        }
        
        // check if have access to view or edit
        if($unit->owner_id != $owner_id){
            return response()->json(config('constants.messages.invalid_access'), 400);
            exit();
        }
            
        foreach($data as $d){
            $id = $d['id'];
            $order_number = $d['order_number'];
            UnitsBoard::where(['id'=>$id, 'unit_id'=>$unit_id])->update(['order_number'=>$order_number]);
        }
        
        $boards = Unit::find($unit_id)->boards;
        
        return response()->json(['message' => 'Updated Successfully', 'boards' => $boards]);

    }
    
    public function UnitFileFolder(Request $request)
    {
        
        $user_id = auth()->user()->id;

        // check if permission to do
        if(!User::checkPermission('unit-file-folder')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }
        
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'order_number' => 'required|numeric',
            'unit_id' => 'required|numeric',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 400);
        }
        
        $update_criteria = [];
        if(!empty($request->id)){
            $update_criteria['id'] = $request->id;
        }
        else{
            $update_criteria['id'] = 0;
        }
        
        $folder = UnitsFilesFolder::updateOrCreate(
            $update_criteria,
            [
                'name' => $request->name, 
                'picture' => $request->picture ?? '',
                'order_number' => $request->order_number, 
                'parent_id' => $request->parent_id ?? null,
                'unit_id' => $request->unit_id,
                'unit_manager_sharing' => $request->unit_manager_sharing ?? 0,
                'staff_sharing' => $request->staff_sharing ?? 0,
                'tenant_sharing' => $request->tenant_sharing ?? 0,
            ]
        );

        return response()->json(['message' => 'Success', 'data'=> $folder->id ]);
        
    }
    
    public function getUnitFileFolders(Request $request)
    {
        
        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);

        // check if permission to do
        if(!User::checkPermission('get-unit-file-folders')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }
        
        $unit_id = $request->query('unit_id');
        $unit = Unit::find($unit_id);
        
        // if($unit->owner_id != $owner_id){
        //     return response()->json(config('constants.messages.invalid_access'), 400);
        //     exit();
        // }
        
        $folders = UnitsFilesFolder::select(['*', 'name as title'])->where(['unit_id' => $unit_id])->get();
        
        if(!empty($folders)){
            $folders = $folders->toArray();
            $folders = $this->generic->parentChildTree($folders);
        }
        
        return response()->json(['message' => 'Success', 'data'=> ['folders' => $folders] ]);
        
    }
    
    public function deleteUnitFileFolder(Request $request){

        // check if permission to do
        if(!User::checkPermission('delete-unit-file-folder')){
            return response()->json(config('constants.messages.permission_denied'), 400);
            exit();
        }

        $user_id = auth()->user()->id;
        $owner_id = User::getOwnerOfUser($user_id);
        
        $folder_id = $request->query('folder_id');
        $folder = UnitsFilesFolder::find($folder_id);
        
        // check if folder exists
        if(empty($folder)){
            return response()->json(['message' => 'No folder found']);
        }
        
        $unit_id = $folder->unit_id;
        
        $unit = Unit::find($unit_id);
        
        // check if have access to view or edit
        if($unit->owner_id != $owner_id){
            return response()->json(config('constants.messages.invalid_access'), 400);
            exit();
        }
        
        UnitsFilesFolder::find($folder_id)->delete();
        
        return response()->json(['message' => 'Deleted', 'data'=> [] ]);

    }

}